package com.useasy.callsdk.demo

import android.app.Activity
import android.bluetooth.BluetoothAdapter
import android.bluetooth.BluetoothHeadset
import android.bluetooth.BluetoothProfile
import android.content.Intent
import android.content.IntentFilter
import android.media.AudioManager
import android.os.Bundle
import android.os.PowerManager
import android.os.SystemClock
import android.util.DisplayMetrics
import android.util.Log
import android.view.Gravity
import android.view.View
import android.view.ViewGroup
import android.view.WindowManager
import android.widget.Chronometer
import android.widget.ImageButton
import android.widget.LinearLayout
import android.widget.RelativeLayout
import android.widget.TextView
import android.widget.Toast
import androidx.core.content.ContextCompat
import com.useasy.callsdk.CallSDK
import com.useasy.callsdk.bean.SipParams
import com.useasy.callsdk.listener.CallStateListener
import com.useasy.callsdk.utils.AudioUtils
import com.useasy.callsdk.utils.Loger
import org.json.JSONObject

class CallActivity : BaseActivity() {

    companion object {
        private const val TAG = "CallActivity"

        fun getLinearLayout(activity: Activity): LinearLayout {
            val layout = LinearLayout(activity)
            val params = LinearLayout.LayoutParams(
                ViewGroup.LayoutParams.WRAP_CONTENT, ViewGroup.LayoutParams.WRAP_CONTENT
            )
            layout.layoutParams = params
            layout.orientation = LinearLayout.HORIZONTAL
            return layout
        }
    }

    private lateinit var mActivityMain: RelativeLayout
    private lateinit var mTvPhone: TextView
    private lateinit var mTvCallDurtionPhone: TextView
    private lateinit var mIbtnHangUpAm: ImageButton
    private lateinit var mTvMuteAm: TextView
    private lateinit var mTvDailAm: TextView
    private lateinit var mTvSpeakerAm: TextView
    private lateinit var mLlDailLayout: LinearLayout
    private lateinit var mTvHiddenDail: TextView
    private lateinit var mLlOptionAm: LinearLayout
    private lateinit var mChronometer: Chronometer


    private var phoneNumber: String = ""
    private var mSipParams: SipParams? = null
    private var wakeLock: PowerManager.WakeLock? = null
    private var hasChangedSpeaker: Boolean = false
    private var mHeadsetPlugReceiver: HeadsetPlugReceiver? = null
    private var bluetoothHeadset: BluetoothHeadset? = null
    private var bluetoothAdapter: BluetoothAdapter? = null

    private val dailNumber = arrayOf("1", "2", "3", "4", "5", "6", "7", "8", "9", "*", "0", "#")

    private val itemOnClickListener = View.OnClickListener { v ->
        when (v.id) {
            R.id.ibtn_hang_up_am -> {
                CallSDK.hangupCall()
                finish()
            }

            R.id.tv_mute_am -> {
                if (CallSDK.getMuteStatus()) {
                    CallSDK.openMute(false)
                    setMuteTopDrawable(R.drawable.callsdk_mute_unenable)
                    mTvMuteAm.setTextColor(
                        ContextCompat.getColor(
                            this, R.color.callsdk_color_e6e6e6
                        )
                    )
                } else {
                    CallSDK.openMute(true)
                    setMuteTopDrawable(R.drawable.callsdk_mute_enable)
                    mTvMuteAm.setTextColor(ContextCompat.getColor(this, R.color.callsdk_white))
                }
            }

            R.id.tv_dail_am -> {
                showDailView()
            }

            R.id.tv_speaker_am -> {
                mTvCallDurtionPhone.post {
                    hasChangedSpeaker = true
                    switchSpeaker()
                }
            }

            R.id.tv_hidden_dail -> {
                mLlDailLayout.visibility = View.GONE
                mTvHiddenDail.visibility = View.GONE
                mLlOptionAm.visibility = View.VISIBLE
            }
        }
    }

    private val dialOnClickListener = View.OnClickListener { view ->
        val tag = view.tag as? String ?: return@OnClickListener
        val phone = mTvPhone.text.toString().trim()
        mTvPhone.text = phone + tag
        sendTMS(tag)
    }


    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.callsdk_activity_call)
        resolveIntent()
        bindViews()
        initmHeadsetPlugReceiver()

        val powerManager = getSystemService(POWER_SERVICE) as PowerManager
        wakeLock =
            powerManager.newWakeLock(PowerManager.PARTIAL_WAKE_LOCK, "MicrophoneApp:WakeLock")
        wakeLock?.acquire()
        window.addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)

        startCallService()

        val userData = JSONObject().apply {
            put("test", "test user data")
        }
        CallSDK.makeCall(phoneNumber, userData, object : CallStateListener {
            override fun onConnected() {
                super.onConnected()
                Loger.d(TAG, "onConnected")
            }

            override fun onDisconnected() {
                super.onDisconnected()
                Loger.d(TAG, "onDisconnected")
            }

            override fun onCallAnswered(callId: String?) {
                super.onCallAnswered(callId)
                Loger.d(TAG, "onCallAnswered: $callId")
                val currentRoute = getCurrentAudioRoute()
                AudioUtils.choiceAudioModel(this@CallActivity)
                updateSpeakerUI(currentRoute)

                mTvCallDurtionPhone.visibility = View.GONE
                mChronometer.visibility = View.VISIBLE
                mChronometer.base = SystemClock.elapsedRealtime()
                mChronometer.start()
            }

            override fun onCallProceeding(callId: String?) {
                super.onCallProceeding(callId)
                Loger.d(TAG, "onCallProceeding: $callId")
            }

            override fun onCallReleased(callId: String?) {
                super.onCallReleased(callId)
                Loger.d(TAG, "onCallReleased: $callId")
                if (getSpeakerStatus()) {
                    switchSpeaker()
                }
                stopTimeCountDown()
                finish()
            }

            override fun onCallFailed(callId: String?, errorCode: Int, errorMsg: String?) {
                super.onCallFailed(callId, errorCode, errorMsg)
                Loger.d(TAG, "onCallFailed: callId=$callId, errorCode=$errorCode, errorMsg=$errorMsg")
                Toast.makeText(this@CallActivity, errorMsg, Toast.LENGTH_LONG).show()
                if (getSpeakerStatus()) {
                    switchSpeaker()
                }
                stopTimeCountDown()
                finish()
            }

            override fun onCallAlerting(callId: String?) {
                super.onCallAlerting(callId)
                mTvCallDurtionPhone.setText(R.string.callsdk_use_calling)
            }

            override fun onDtmfReceived(callId: String?, dtmf: Char) {
                super.onDtmfReceived(callId, dtmf)
                Loger.d(TAG, "onDtmfReceived: $callId, dtmf: $dtmf")
            }
        })
    }

    private fun resolveIntent() {
        phoneNumber = intent.getStringExtra("phoneNumber").toString()
    }

    private fun bindViews() {
        mActivityMain = findViewById(R.id.activity_main)
        mTvPhone = findViewById(R.id.tv_contact_phone)
        mTvCallDurtionPhone = findViewById(R.id.tv_call_durtion_phone)
        mIbtnHangUpAm = findViewById(R.id.ibtn_hang_up_am)
        mTvMuteAm = findViewById(R.id.tv_mute_am)
        mTvDailAm = findViewById(R.id.tv_dail_am)
        mTvSpeakerAm = findViewById(R.id.tv_speaker_am)
        mLlDailLayout = findViewById(R.id.ll_dail_layout)
        mTvHiddenDail = findViewById(R.id.tv_hidden_dail)
        mLlOptionAm = findViewById(R.id.ll_option_am)
        mChronometer = findViewById(R.id.chronometer_am)

        mIbtnHangUpAm.setOnClickListener(itemOnClickListener)
        mTvMuteAm.setOnClickListener(itemOnClickListener)
        mTvDailAm.setOnClickListener(itemOnClickListener)
        mTvSpeakerAm.setOnClickListener(itemOnClickListener)
        mTvHiddenDail.setOnClickListener(itemOnClickListener)
        if (phoneNumber.isNotEmpty()) {
            mTvPhone.text = phoneNumber
        } else {
            mSipParams?.let {
                mTvPhone.text = it.getPhoneNum() ?: ""
            }
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        stopCallService()
        unregisterHeadsetPlugReceiver()
        releaseBluetoothProfile()
        wakeLock?.release()
        window.clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)
    }
    
    /**
     * 释放蓝牙 Profile
     */
    private fun releaseBluetoothProfile() {
        bluetoothHeadset?.let {
            try {
                bluetoothAdapter?.closeProfileProxy(BluetoothProfile.HEADSET, it)
                Loger.dTag(TAG, "BluetoothHeadset profile released")
            } catch (e: Exception) {
                Loger.eTag(TAG, "释放蓝牙 Profile 失败: ${e.message}")
            }
            bluetoothHeadset = null
        }
    }

    /**
     * 初始化耳机插拔监听器
     */
    private fun initmHeadsetPlugReceiver() {
        mHeadsetPlugReceiver = HeadsetPlugReceiver(this, object : AudioRouteChangeListener {
            override fun onAudioRouteChanged(route: AudioRoute) {
                updateSpeakerUI(route)
            }
        })
        val intentFilter = IntentFilter().apply {
            addAction(Intent.ACTION_HEADSET_PLUG)
            addAction(BluetoothHeadset.ACTION_CONNECTION_STATE_CHANGED)
            addAction(AudioManager.ACTION_AUDIO_BECOMING_NOISY)
        }
        registerReceiver(mHeadsetPlugReceiver, intentFilter)
        Loger.dTag(TAG, "HeadsetPlugReceiver registered")
        
        initBluetoothProfile()
    }
    
    /**
     * 获取当前音频路由
     */
    private fun getCurrentAudioRoute(): AudioRoute {
        return when {
            AudioUtils.isWiredHeadsetOn(this) -> AudioRoute.RECEIVER
            AudioUtils.isBluetoothA2dpOn(this) -> AudioRoute.BLUETOOTH
            else -> AudioRoute.SPEAKER
        }
    }
    
    /**
     * 根据音频路由更新扬声器UI
     */
    private fun updateSpeakerUI(route: AudioRoute) {
        when (route) {
            AudioRoute.SPEAKER -> {
                // 外放模式：显示扬声器开启状态
                setSpeakerTopDrawable(R.drawable.callsdk_speaker_enable)
                mTvSpeakerAm.setTextColor(ContextCompat.getColor(this, R.color.callsdk_white))
            }
            AudioRoute.RECEIVER -> {
                // 听筒模式：显示扬声器关闭状态
                setSpeakerTopDrawable(R.drawable.callsdk_speaker_unenable)
                mTvSpeakerAm.setTextColor(ContextCompat.getColor(this, R.color.callsdk_color_e6e6e6))
            }
            AudioRoute.BLUETOOTH -> {
                // 蓝牙模式：显示扬声器关闭状态（因为音频通过蓝牙输出）
                setSpeakerTopDrawable(R.drawable.callsdk_speaker_unenable)
                mTvSpeakerAm.setTextColor(ContextCompat.getColor(this, R.color.callsdk_color_e6e6e6))
            }
        }
        Loger.dTag(TAG, "扬声器UI已更新: $route")
    }
    
    /**
     * 初始化蓝牙 Profile，以便接收蓝牙连接状态变化广播
     */
    private fun initBluetoothProfile() {
        try {
            bluetoothAdapter = BluetoothAdapter.getDefaultAdapter()
            bluetoothAdapter?.getProfileProxy(
                this,
                object : BluetoothProfile.ServiceListener {
                    override fun onServiceConnected(profile: Int, proxy: BluetoothProfile) {
                        if (profile == BluetoothProfile.HEADSET) {
                            bluetoothHeadset = proxy as BluetoothHeadset
                            Loger.dTag(TAG, "BluetoothHeadset profile connected")
                            // 检查当前连接状态
                            val connectedDevices = bluetoothHeadset?.connectedDevices
                            if (connectedDevices != null && connectedDevices.isNotEmpty()) {
                                Loger.dTag(TAG, "蓝牙耳机已连接，设备数量: ${connectedDevices.size}")
                                AudioUtils.changeToHeadset(this@CallActivity)
                                runOnUiThread {
                                    updateSpeakerUI(AudioRoute.BLUETOOTH)
                                }
                            }
                        }
                    }

                    override fun onServiceDisconnected(profile: Int) {
                        if (profile == BluetoothProfile.HEADSET) {
                            Loger.dTag(TAG, "BluetoothHeadset profile disconnected")
                            bluetoothHeadset = null
                        }
                    }
                },
                BluetoothProfile.HEADSET
            )
        } catch (e: Exception) {
            Loger.eTag(TAG, "初始化蓝牙 Profile 失败: ${e.message}")
        }
    }

    /**
     * 注销耳机插拔监听器
     */
    private fun unregisterHeadsetPlugReceiver() {
        mHeadsetPlugReceiver?.let {
            try {
                unregisterReceiver(it)
                Loger.dTag(TAG, "HeadsetPlugReceiver unregistered")
            } catch (e: Exception) {
                Loger.eTag(TAG, "Failed to unregister HeadsetPlugReceiver: ${e.message}")
            }
            mHeadsetPlugReceiver = null
        }
    }

    private fun startCallService() {
        val intent = Intent(this, CallService::class.java).apply {
            action = CallService.ACTION_START_CALL
            putExtra(CallService.EXTRA_PHONE_NUMBER, phoneNumber)
        }
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O) {
            startForegroundService(intent)
        } else {
            startService(intent)
        }
        Loger.d(TAG, "CallService started")
    }

    private fun stopCallService() {
        val intent = Intent(this, CallService::class.java).apply {
            action = CallService.ACTION_STOP_CALL
        }
        stopService(intent)
        Loger.d(TAG, "CallService stopped")
    }

    private fun setSpeakerTopDrawable(resId: Int) {
        val topDrawable = ContextCompat.getDrawable(this, resId)
        topDrawable?.setBounds(0, 0, topDrawable.minimumWidth, topDrawable.minimumHeight)
        mTvSpeakerAm.setCompoundDrawables(null, topDrawable, null, null)
    }

    private fun setMuteTopDrawable(resId: Int) {
        val topDrawable = ContextCompat.getDrawable(this, resId)
        topDrawable?.setBounds(0, 0, topDrawable.minimumWidth, topDrawable.minimumHeight)
        mTvMuteAm.setCompoundDrawables(null, topDrawable, null, null)
    }

    private fun stopTimeCountDown() {
        if (mChronometer.visibility == View.VISIBLE) {
            mChronometer.stop()
        }
    }

    private fun showDailView() {
        mLlDailLayout.visibility = View.VISIBLE
        mTvHiddenDail.visibility = View.VISIBLE
        mLlOptionAm.visibility = View.GONE
        if (mLlDailLayout.childCount > 2) {
            return
        }
        var layout: LinearLayout? = null
        for (i in dailNumber.indices) {
            if (i % 3 == 0) {
                layout = getLinearLayout(this)
                mLlDailLayout.addView(layout)
            }
            val textView = getItemView(dailNumber[i])
            layout?.addView(textView)
        }
    }

    private fun getItemView(s: String): TextView {
        val textView = TextView(this)
        val screenWidth = getDisplay(this).widthPixels
        val params = LinearLayout.LayoutParams(screenWidth / 5, screenWidth / 5)
        params.setMargins(screenWidth / 30, screenWidth / 50, screenWidth / 30, 0)
        textView.layoutParams = params
        textView.gravity = Gravity.CENTER
        textView.textSize = (screenWidth / 40).toFloat()
        textView.text = s
        textView.setBackgroundResource(R.drawable.callsdk_bg_circle_white)
        textView.setTextColor(ContextCompat.getColor(this, R.color.callsdk_maincolor))
        textView.tag = s
        textView.setOnClickListener(dialOnClickListener)
        return textView
    }

    private fun sendTMS(tag: String) {
        CallSDK.sendDTMF(tag[0].toString())
    }

    /**
     * 返回SDK免提状态
     *
     * @return 免提状态 true:外放 false:听筒
     */
    @Synchronized
    fun getSpeakerStatus(): Boolean {
        return CallSDK.isLoudSpeakerOn()
    }

    /**
     * 切换SDK免提状态
     */
    @Synchronized
    fun switchSpeaker() {
        switchSpeaker(!getSpeakerStatus())
    }

    /**
     * 切换SDK免提状态
     */
    @Synchronized
    fun switchSpeaker(mSpeaker: Boolean) {
        Loger.d(TAG, "switchSpeaker speaker: $mSpeaker")
        CallSDK.openLoudSpeaker(mSpeaker)
        if (!mSpeaker) {
            setSpeakerTopDrawable(R.drawable.callsdk_speaker_unenable)
            mTvSpeakerAm.setTextColor(ContextCompat.getColor(this, R.color.callsdk_color_e6e6e6))
        } else {
            setSpeakerTopDrawable(R.drawable.callsdk_speaker_enable)
            mTvSpeakerAm.setTextColor(ContextCompat.getColor(this, R.color.callsdk_white))
        }
    }

    fun getDisplay(activity: Activity): DisplayMetrics {
        val manager = activity.windowManager
        val outMetrics = DisplayMetrics()
        manager.defaultDisplay.getMetrics(outMetrics)
        return outMetrics
    }

}