//
//  ActiveCallDialKeyboard.swift
//  JJPhoneDemo
//
//  通话页面的数字键盘 - 用于发送 DTMF 信号

import UIKit

protocol ActiveCallDialKeyboardDelegate: AnyObject {
    func activeCallDialKeyboardDidInputNumber(_ number: String)
}

class ActiveCallDialKeyboard: UIView {
    
    weak var delegate: ActiveCallDialKeyboardDelegate?
    
    private var numberButtons: [UIButton] = []
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        setupUI()
    }
    
    convenience init() {
        self.init(frame: .zero)
    }
    
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    private func setupUI() {
        backgroundColor = .clear // 透明背景，与黑色背景融合
        
        // 创建 12 个按钮：1-9, *, 0, #
        let numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "*", "0", "#"]
        
        for (index, number) in numbers.enumerated() {
            let button = createButton(number: number, tag: index)
            numberButtons.append(button)
            addSubview(button)
        }
        
        setupConstraints()
    }
    
    private func createButton(number: String, tag: Int) -> UIButton {
        let button = UIButton(type: .custom)
        button.tag = tag
        button.setTitle(number, for: .normal)
        if #available(iOS 13.0, *) {
            button.titleLabel?.font = .systemFont(ofSize: 32, weight: .medium)
        } else {
            button.titleLabel?.font = .systemFont(ofSize: 32, weight: .semibold)
        }
        button.setTitleColor(.white, for: .normal)
        
        // 优化按钮背景：使用半透明深色，与黑色背景更搭配
        // 正常状态：深灰半透明
        button.backgroundColor = UIColor(white: 0.2, alpha: 0.6)
        button.layer.cornerRadius = 35 // 圆形按钮
        button.layer.borderWidth = 1
        button.layer.borderColor = UIColor(white: 0.3, alpha: 0.3).cgColor
        
        // 添加阴影效果，增强层次感
        button.layer.shadowColor = UIColor.black.cgColor
        button.layer.shadowOffset = CGSize(width: 0, height: 2)
        button.layer.shadowRadius = 4
        button.layer.shadowOpacity = 0.3
        
        button.translatesAutoresizingMaskIntoConstraints = false
        button.addTarget(self, action: #selector(buttonTapped(_:)), for: .touchUpInside)
        
        return button
    }
    
    private func setupConstraints() {
        let buttonSize: CGFloat = 70
        let screenWidth = UIScreen.main.bounds.width
        let horizontalSpacing = (screenWidth - buttonSize * 3) / 4
        let verticalSpacing: CGFloat = 20
        
        for (index, button) in numberButtons.enumerated() {
            let row = index / 3
            let col = index % 3
            
            NSLayoutConstraint.activate([
                button.widthAnchor.constraint(equalToConstant: buttonSize),
                button.heightAnchor.constraint(equalToConstant: buttonSize),
                button.leadingAnchor.constraint(
                    equalTo: leadingAnchor,
                    constant: horizontalSpacing + CGFloat(col) * (buttonSize + horizontalSpacing)
                ),
                button.topAnchor.constraint(
                    equalTo: topAnchor,
                    constant: CGFloat(row) * (buttonSize + verticalSpacing)
                )
            ])
        }
    }
    
    @objc private func buttonTapped(_ button: UIButton) {
        guard let number = button.currentTitle else { return }
        
        // 按钮点击反馈动画：更明显的视觉反馈
        UIView.animate(withDuration: 0.1, animations: {
            button.transform = CGAffineTransform(scaleX: 0.9, y: 0.9)
            button.backgroundColor = UIColor(white: 0.4, alpha: 0.8)
        }) { _ in
            UIView.animate(withDuration: 0.2) {
                button.transform = .identity
                button.backgroundColor = UIColor(white: 0.2, alpha: 0.6)
            }
        }
        
        delegate?.activeCallDialKeyboardDidInputNumber(number)
    }
}


