//
//  AppDelegate.swift
//  JJPhoneDemo
//
//

import UIKit
import AVFoundation

@main
class AppDelegate: UIResponder, UIApplicationDelegate {

    var window: UIWindow?

    func application(_ application: UIApplication, didFinishLaunchingWithOptions launchOptions: [UIApplication.LaunchOptionsKey: Any]?) -> Bool {
        // 在应用启动时立即配置 AVAudioSession（在 PJSIP 初始化之前）
        configureAudioSessionEarly()
        
        // 配置 VoIP 日志等级（必须在初始化之前设置）
        configureVoIPLogging()
        
        // 可选：提前初始化 VoIP（也可以延迟到登录时）
        JJPhoneSDK.shared.initializeIfNeeded()
        
        window = UIWindow(frame: UIScreen.main.bounds)
        if #available(iOS 13.0, *) {
            window?.backgroundColor = .systemBackground
        } else {
            window?.backgroundColor = .white
        }
        
        // 使用 SDK 提供的 UI 作为入口
        let phoneVC = JJPhoneSDK.shared.makePhoneMainViewController()
        let nav = UINavigationController(rootViewController: phoneVC)
        
        window?.rootViewController = nav
        window?.makeKeyAndVisible()
        
        return true
    }
    
    /// 在应用启动时提前配置 AVAudioSession
    private func configureAudioSessionEarly() {
        do {
            let audioSession = AVAudioSession.sharedInstance()
            
            // 配置音频会话类别（默认使用听筒模式，不设置 .defaultToSpeaker）
            try audioSession.setCategory(
                .playAndRecord,
                mode: .voiceChat,
                options: [.allowBluetooth, .allowBluetoothA2DP, .duckOthers]
            )
            
            // 激活音频会话
            try audioSession.setActive(true, options: [])
            
            // 请求麦克风权限（异步，不阻塞启动）
            audioSession.requestRecordPermission { granted in
                if !granted {
                    AppLogger.warning("Microphone permission denied", category: "AppDelegate")
                }
            }
        } catch {
            AppLogger.error("AVAudioSession configuration failed - \(error.localizedDescription)", category: "AppDelegate")
        }
    }
    
    /// 配置 VoIP 日志系统
    private func configureVoIPLogging() {
        // 设置日志等级为 Verbose (2)，输出所有日志（业务、SIP、错误）
        VoIPManager.setLogLevel(2)
        
        // 启用文件日志
        VoIPManager.setFileLoggingEnabled(true)
        
        // 使用默认日志路径：Documents/VoIPLogs/voip.log
        // PJSIP 日志会自动生成在同一目录：Documents/VoIPLogs/voip_pjsip.log
    }

}

